<?php
header("Content-Type: application/json; charset=UTF-8");
header("Access-Control-Allow-Origin: *");
header("Access-Control-Allow-Methods: GET, OPTIONS");
header("Access-Control-Allow-Headers: Content-Type, Authorization, X-Requested-With");

// Handle preflight OPTIONS request
if ($_SERVER['REQUEST_METHOD'] == 'OPTIONS') {
    http_response_code(200);
    exit();
}

require_once __DIR__ . "/../db.php";
require_once __DIR__ . '/../session.php';

$method = $_SERVER['REQUEST_METHOD'];
$action = $_GET['action'] ?? '';

// Helper function for response
function sendResponse($status, $message, $data = null, $code = 200) {
    http_response_code($code);
    echo json_encode([
        "status" => $status,
        "message" => $message,
        "data" => $data
    ]);
    exit;
}

// Auth guard for all non-OPTIONS requests
if (!validateApiToken()) {
    sendResponse(false, "Authentication required", null, 401);
}

$business_id = (int)($_SESSION['business_id'] ?? 0);

try {
    if ($method === 'GET') {
        
        if ($action === 'sales' || $action === '') {
            $from = $_GET['from'] ?? date('Y-m-01');
            $to = $_GET['to'] ?? date('Y-m-d');
            $limit = intval($_GET['limit'] ?? 1000);
            
            // Fetch paid orders in range
            $orders = all_rows("
                SELECT o.*, c.name AS customer_name, c.mobile_no AS customer_mobile
                FROM orders o
                LEFT JOIN customer c ON o.customer_sno = c.sno
                WHERE o.business_id = ? AND DATE(o.created_at) BETWEEN ? AND ?
                  AND o.status = 'PAID'
                ORDER BY o.created_at DESC
                LIMIT ?
            ", [$business_id, $from, $to, $limit]);
            
            // Calculate summary
            $summary = [
                'order_count' => count($orders),
                'total_revenue' => 0.0,
                'total_subtotal' => 0.0,
                'total_discount' => 0.0,
                'total_tax' => 0.0
            ];
            
            $payment_methods = [];
            
            foreach ($orders as $order) {
                $summary['total_revenue'] += (float)$order['total'];
                $summary['total_subtotal'] += (float)$order['subtotal'];
                $summary['total_discount'] += (float)$order['discount'];
                $summary['total_tax'] += (float)$order['tax'];
                
                $pm = $order['payment_method'] ?: 'UNKNOWN';
                if (!isset($payment_methods[$pm])) {
                    $payment_methods[$pm] = ['count' => 0, 'amount' => 0.0];
                }
                $payment_methods[$pm]['count']++;
                $payment_methods[$pm]['amount'] += (float)$order['total'];
            }
            
            $report_data = [
                'orders' => $orders,
                'summary' => $summary,
                'payment_methods' => $payment_methods,
                'date_range' => ['from' => $from, 'to' => $to]
            ];
            
            sendResponse(true, "Sales report generated successfully", $report_data);
        }
        
        elseif ($action === 'items') {
            $from = $_GET['from'] ?? date('Y-m-01');
            $to = $_GET['to'] ?? date('Y-m-d');
            
            // Top selling items
            $top_items = all_rows("SELECT 
                                    oi.item_name, 
                                    SUM(oi.qty) as total_qty,
                                    SUM(oi.price * oi.qty) as total_amount,
                                    COUNT(DISTINCT oi.order_id) as order_count,
                                    AVG(oi.price) as avg_price
                                  FROM order_items oi
                                  INNER JOIN orders o ON oi.order_id = o.id
                                  WHERE o.status = 'PAID' AND o.business_id = ?
                                    AND DATE(o.created_at) BETWEEN ? AND ?
                                  GROUP BY oi.item_name
                                  ORDER BY total_qty DESC", [$business_id, $from, $to]);
            
            sendResponse(true, "Item sales report generated successfully", $top_items);
        }
        
        elseif ($action === 'customers') {
            $from = $_GET['from'] ?? date('Y-m-01');
            $to = $_GET['to'] ?? date('Y-m-d');
            
            // Customer statistics
            $customer_stats = all_rows("SELECT 
                                        c.name as customer_name,
                                        c.mobile_no as customer_mobile,
                                        COUNT(o.id) as total_orders,
                                        SUM(o.total) as total_spent,
                                        MAX(o.created_at) as last_order_date
                                      FROM customer c
                                      INNER JOIN orders o ON c.sno = o.customer_sno
                                      WHERE o.status = 'PAID' AND o.business_id = ?
                                        AND DATE(o.created_at) BETWEEN ? AND ?
                                      GROUP BY c.sno
                                      ORDER BY total_spent DESC", [$business_id, $from, $to]);
            
            sendResponse(true, "Customer report generated successfully", $customer_stats);
        }
        
        elseif ($action === 'daily') {
            $from = $_GET['from'] ?? date('Y-m-01');
            $to = $_GET['to'] ?? date('Y-m-d');
            
            // Daily sales breakdown
            $daily_sales = all_rows("SELECT 
                                      DATE(created_at) as sale_date,
                                      COUNT(*) as order_count,
                                      SUM(total) as total_revenue,
                                      SUM(subtotal) as total_subtotal,
                                      SUM(discount) as total_discount,
                                      SUM(tax) as total_tax
                                    FROM orders 
                                    WHERE status = 'PAID' AND business_id = ?
                                      AND DATE(created_at) BETWEEN ? AND ?
                                    GROUP BY DATE(created_at)
                                    ORDER BY sale_date DESC", [$business_id, $from, $to]);
            
            sendResponse(true, "Daily sales report generated successfully", $daily_sales);
        }
        
        elseif ($action === 'export') {
            $from = $_GET['from'] ?? date('Y-m-01');
            $to = $_GET['to'] ?? date('Y-m-d');
            
            // Fetch orders for CSV export
            $orders = all_rows("
                SELECT o.*, c.name AS customer_name, c.mobile_no AS customer_mobile
                FROM orders o
                LEFT JOIN customer c ON o.customer_sno = c.sno
                WHERE o.business_id = ? AND DATE(o.created_at) BETWEEN ? AND ?
                  AND o.status = 'PAID'
                ORDER BY o.created_at DESC
            ", [$business_id, $from, $to]);
            
            // Set CSV headers
            header('Content-Type: text/csv');
            header('Content-Disposition: attachment; filename="sales_report_' . $from . '_to_' . $to . '.csv"');
            
            $output = fopen('php://output', 'w');
            
            // Add CSV headers
            fputcsv($output, ['Order ID', 'Date', 'Table No', 'Customer Name', 'Customer Mobile', 'Payment Method', 'Subtotal', 'Discount', 'Tax', 'Total']);
            
            foreach ($orders as $order) {
                fputcsv($output, [
                    $order['id'],
                    $order['created_at'],
                    $order['table_no'],
                    $order['customer_name'] ?: '',
                    $order['customer_mobile'] ?: '',
                    $order['payment_method'] ?: '',
                    $order['subtotal'],
                    $order['discount'],
                    $order['tax'],
                    $order['total']
                ]);
            }
            
            fclose($output);
            exit;
        }
        
        else {
            sendResponse(false, "Invalid action", null, 400);
        }
    }
    
    else {
        sendResponse(false, "Method not allowed", null, 405);
    }
    
} catch (Exception $e) {
    sendResponse(false, "Server error: " . $e->getMessage(), null, 500);
}
?>
