<?php
header("Content-Type: application/json; charset=UTF-8");
header("Access-Control-Allow-Origin: *");
header("Access-Control-Allow-Methods: GET, OPTIONS");
header("Access-Control-Allow-Headers: Content-Type, Authorization, X-Requested-With");

// Handle preflight OPTIONS request
if ($_SERVER['REQUEST_METHOD'] == 'OPTIONS') {
    http_response_code(200);
    exit();
}

require_once __DIR__ . "/../db.php";
require_once __DIR__ . '/../session.php';

$method = $_SERVER['REQUEST_METHOD'];
$action = $_GET['action'] ?? '';

// Helper function for response
function sendResponse($status, $message, $data = null, $code = 200) {
    http_response_code($code);
    echo json_encode([
        "status" => $status,
        "message" => $message,
        "data" => $data
    ]);
    exit;
}

// Auth guard for all non-OPTIONS requests
if (!validateApiToken()) {
    sendResponse(false, "Authentication required", null, 401);
}

$business_id = (int)($_SESSION['business_id'] ?? 0);

try {
    if ($method === 'GET') {
        
        if ($action === 'get' || $action === '') {
            $order_id = intval($_GET['id'] ?? 0);
            
            if ($order_id <= 0) {
                sendResponse(false, "Invalid order ID", null, 400);
            }
            
            // Get order details
            $order = one("SELECT o.*, c.name AS customer_name, c.mobile_no AS customer_mobile
                         FROM orders o
                         LEFT JOIN customer c ON o.customer_sno = c.sno
                         WHERE o.id = ? AND o.business_id = ?", [$order_id, $business_id]);
            
            if (!$order) {
                sendResponse(false, "Order not found", null, 404);
            }
            
            // Get settings
            $settings = one("SELECT * FROM settings WHERE id = ?", [$business_id]);
            if (!$settings) {
                $settings = [
                    'business_name' => 'Restaurant',
                    'address' => '',
                    'phone' => '',
                    'gstin' => ''
                ];
            }
            
            // Get order items
            $items = all_rows("SELECT * FROM order_items WHERE order_id = ? ORDER BY id", [$order_id]);
            
            // Calculate totals
            $subtotal = 0.0;
            foreach ($items as $item) {
                $subtotal += $item['price'] * $item['qty'];
            }
            
            $discount = (float)$order['discount'];
            $tax = (float)$order['tax'];
            $total = (float)$order['total'];
            
            $receipt_data = [
                'order' => $order,
                'settings' => $settings,
                'items' => $items,
                'totals' => [
                    'subtotal' => $subtotal,
                    'discount' => $discount,
                    'tax' => $tax,
                    'total' => $total
                ]
            ];
            
            sendResponse(true, "Receipt data fetched successfully", $receipt_data);
        }
        
        elseif ($action === 'print') {
            $order_id = intval($_GET['id'] ?? 0);
            
            if ($order_id <= 0) {
                sendResponse(false, "Invalid order ID", null, 400);
            }
            
            // Get order details
            $order = one("SELECT o.*, c.name AS customer_name, c.mobile_no AS customer_mobile
                         FROM orders o
                         LEFT JOIN customer c ON o.customer_sno = c.sno
                         WHERE o.id = ?", [$order_id]);
            
            if (!$order) {
                sendResponse(false, "Order not found", null, 404);
            }
            
            // Get settings
            $settings = one("SELECT * FROM settings WHERE id = 1");
            if (!$settings) {
                $settings = [
                    'business_name' => 'Restaurant',
                    'address' => '',
                    'phone' => '',
                    'gstin' => ''
                ];
            }
            
            // Get order items
            $items = all_rows("SELECT * FROM order_items WHERE order_id = ? ORDER BY id", [$order_id]);
            
            // Calculate totals
            $subtotal = 0.0;
            foreach ($items as $item) {
                $subtotal += $item['price'] * $item['qty'];
            }
            
            $discount = (float)$order['discount'];
            $tax = (float)$order['tax'];
            $total = (float)$order['total'];
            $gst_enabled = (int)$order['gst_enabled'];
            
            // Generate HTML receipt
            $html = '<!DOCTYPE html>
<html>
<head>
    <meta charset="utf-8">
    <title>Receipt #' . $order['id'] . '</title>
    <style>
        body {
            font-family: "Courier New", monospace;
            max-width: 280px;
            margin: auto;
            font-size: 14px;
            background: white;
            padding: 10px;
        }
        h3 {
            text-align: center;
            margin: 5px 0;
            font-size: 18px;
        }
        .center { text-align: center; }
        hr { border: 1px dashed #000; margin: 6px 0; }
        table { width: 100%; border-collapse: collapse; font-size: 13px; }
        th, td { padding: 2px 0; }
        th { text-align: left; border-bottom: 1px dashed #000; }
        td:last-child, th:last-child { text-align: right; }
        .total { font-weight: bold; border-top: 1px dashed #000; padding-top: 5px; }
        .footer { text-align: center; margin-top: 8px; font-size: 13px; }
    </style>
</head>
<body>
    <h3>' . htmlspecialchars($settings['business_name']) . '</h3>
    <div class="center">
        ' . htmlspecialchars($settings['address']) . '<br>
        Ph: ' . htmlspecialchars($settings['phone']) . '<br>';
            
            if (!empty($settings['gstin'])) {
                $html .= 'GSTIN: ' . htmlspecialchars($settings['gstin']) . '<br>';
            }
            
            $html .= '</div>
    <hr>
    <div>
        Receipt: #' . $order['id'] . ' | ' . date("Y-m-d H:i", strtotime($order['created_at'])) . '
    </div>
    <hr>
    <table>
        <tr>
            <th>Item</th>
            <th>Qty</th>
            <th>Total</th>
        </tr>';
            
            foreach ($items as $item) {
                $html .= '<tr>
                    <td>' . htmlspecialchars($item['item_name']) . '</td>
                    <td>' . $item['qty'] . '</td>
                    <td>₹' . number_format($item['price'] * $item['qty'], 2) . '</td>
                </tr>';
            }
            
            $html .= '</table>
    <hr>
    <table>
        <tr><td>Subtotal</td><td>₹' . number_format($subtotal, 2) . '</td></tr>
        <tr><td>Discount</td><td>-₹' . number_format($discount, 2) . '</td></tr>';
            
            if ($gst_enabled) {
                $html .= '<tr><td>GST</td><td>₹' . number_format($tax, 2) . '</td></tr>';
            }
            
            $html .= '<tr class="total"><td>Total</td><td>₹' . number_format($total, 2) . '</td></tr>
    </table>
    <div class="footer">
        Paid via <strong>' . strtoupper($order['payment_method'] ?? 'CASH') . '</strong><br>
        -----------------------------<br>
        ⭐ Thanks! Visit Again ⭐
    </div>
</body>
</html>';
            
            // Return HTML receipt
            sendResponse(true, "Receipt generated successfully", [
                'html' => $html,
                'order_id' => $order_id,
                'total' => $total
            ]);
        }
        
        else {
            sendResponse(false, "Invalid action", null, 400);
        }
    }
    
    else {
        sendResponse(false, "Method not allowed", null, 405);
    }
    
} catch (Exception $e) {
    sendResponse(false, "Server error: " . $e->getMessage(), null, 500);
}
?>
