<?php
header("Content-Type: application/json; charset=UTF-8");
header("Access-Control-Allow-Origin: *");
header("Access-Control-Allow-Methods: GET, POST, PUT, DELETE, OPTIONS");
header("Access-Control-Allow-Headers: Content-Type, Authorization, X-Requested-With");

// Handle preflight OPTIONS request
if ($_SERVER['REQUEST_METHOD'] == 'OPTIONS') {
    http_response_code(200);
    exit();
}

require_once __DIR__ . '/../db.php';
require_once __DIR__ . '/../session.php';

$method = $_SERVER['REQUEST_METHOD'];
$action = $_GET['action'] ?? '';
$input = json_decode(file_get_contents("php://input"), true);

// Helper function for response
function sendResponse($status, $message, $data = null, $code = 200) {
    http_response_code($code);
    echo json_encode([
        "status" => $status,
        "message" => $message,
        "data" => $data
    ]);
    exit;
}

// Auth guard for all non-OPTIONS requests
if (!validateApiToken()) {
    sendResponse(false, "Authentication required", null, 401);
}

$business_id = (int)($_SESSION['business_id'] ?? 0);

// Helper function to recalculate order totals
function recalc_order_totals($order_id) {
    $order = one("SELECT discount FROM orders WHERE id=?", [$order_id]);
    $discount = (float)($order['discount'] ?? 0);

    $settings = one("SELECT gst_applicable, gst_rate FROM settings WHERE id=1");
    $gst_enabled = (int)($settings['gst_applicable'] ?? 0);
    $gst_rate    = (float)($settings['gst_rate'] ?? 0);

    $ois = all_rows("SELECT * FROM order_items WHERE order_id=?", [$order_id]);
    $subtotal = 0.0;
    foreach ($ois as $oi) {
        $subtotal += $oi['price'] * $oi['qty'];
    }
    $taxable = max(0.0, $subtotal - $discount);
    $tax = 0.0;
    if ($gst_enabled && $gst_rate > 0) {
        $tax = round($taxable * ($gst_rate / 100.0), 2);
    }
    $total = round($taxable + $tax, 2);
    query("UPDATE orders 
              SET subtotal=?, discount=?, tax=?, total=? 
            WHERE id=?",
        [$subtotal, $discount, $tax, $total, $order_id]);
}

try {
    switch ($method) {

        // ✅ GET — Fetch orders
        case 'GET':
            if ($action === 'list' || $action === '') {
                $status = $_GET['status'] ?? '';
                $limit = intval($_GET['limit'] ?? 100);
                $from_date = $_GET['from_date'] ?? '';
                $to_date = $_GET['to_date'] ?? '';
                
                $sql = "SELECT o.*, c.name AS customer_name, c.mobile_no AS customer_mobile
                        FROM orders o
                        LEFT JOIN customer c ON o.customer_sno = c.sno
                        WHERE o.business_id = ?";
                $params = [$business_id];
                
                if (!empty($status)) {
                    $sql .= " AND o.status = ?";
                    $params[] = $status;
                }
                
                if (!empty($from_date)) {
                    $sql .= " AND DATE(o.created_at) >= ?";
                    $params[] = $from_date;
                }
                
                if (!empty($to_date)) {
                    $sql .= " AND DATE(o.created_at) <= ?";
                    $params[] = $to_date;
                }
                
                $sql .= " ORDER BY o.created_at DESC LIMIT ?";
                $params[] = $limit;
                
                $orders = all_rows($sql, $params);
                sendResponse(true, "Orders fetched successfully", $orders);
            }
            
            // Get single order with items
            elseif ($action === 'get') {
                $id = intval($_GET['id'] ?? 0);
                if ($id <= 0) {
                    sendResponse(false, "Invalid order ID", null, 400);
                }
                
                $order = one("SELECT o.*, c.name AS customer_name, c.mobile_no AS customer_mobile
                             FROM orders o
                             LEFT JOIN customer c ON o.customer_sno = c.sno
                             WHERE o.id = ? AND o.business_id = ?", [$id, $business_id]);
                if (!$order) {
                    sendResponse(false, "Order not found", null, 404);
                }
                
                $items = all_rows("SELECT * FROM order_items WHERE order_id = ? ORDER BY id", [$id]);
                $order['items'] = $items;
                
                sendResponse(true, "Order fetched successfully", $order);
            }
            
            // Get order summary/stats
            elseif ($action === 'summary') {
                $from_date = $_GET['from_date'] ?? date('Y-m-01');
                $to_date = $_GET['to_date'] ?? date('Y-m-d');
                
                $stats = one("SELECT 
                                COUNT(*) as total_orders,
                                SUM(CASE WHEN status = 'PAID' THEN 1 ELSE 0 END) as paid_orders,
                                SUM(CASE WHEN status = 'OPEN' THEN 1 ELSE 0 END) as open_orders,
                                SUM(CASE WHEN status = 'CANCELLED' THEN 1 ELSE 0 END) as cancelled_orders,
                                SUM(CASE WHEN status = 'PAID' THEN total ELSE 0 END) as total_revenue,
                                SUM(CASE WHEN status = 'PAID' THEN subtotal ELSE 0 END) as total_subtotal,
                                SUM(CASE WHEN status = 'PAID' THEN discount ELSE 0 END) as total_discount,
                                SUM(CASE WHEN status = 'PAID' THEN tax ELSE 0 END) as total_tax
                             FROM orders 
                             WHERE business_id = ? AND DATE(created_at) BETWEEN ? AND ?", [$business_id, $from_date, $to_date]);
                
                sendResponse(true, "Order summary fetched successfully", $stats);
            }
            
            else {
                sendResponse(false, "Invalid action", null, 400);
            }
            break;

        // ✅ POST — Create new order
        case 'POST':
            if ($action === 'create') {
                $order_type = trim($input['order_type'] ?? 'table');
                $table_no = trim($input['table_no'] ?? '');
                $customer_name = trim($input['customer_name'] ?? '');
                $customer_mobile = trim($input['customer_mobile'] ?? '');
                $items = $input['items'] ?? [];

                if (empty($customer_name)) {
                    sendResponse(false, "Customer name is required", null, 400);
                }

                if (empty($items) || !is_array($items)) {
                    sendResponse(false, "At least one item is required", null, 400);
                }

                // Set table number based on order type
                if (strtolower($order_type) === 'takeaway') {
                    $table_no = 'Takeaway';
                }

                // Create customer record
                query("INSERT INTO customer (name, mobile_no, business_id) VALUES (?, ?, ?)", 
                      [$customer_name, $customer_mobile, $business_id]);
                $customer_sno = db()->lastInsertId();

                // Get global GST setting
                $settings = one("SELECT gst_applicable FROM settings WHERE id=1");
                $gst_enabled = (int)($settings['gst_applicable'] ?? 0);

                // Create order
                query("INSERT INTO orders (table_no, status, subtotal, discount, tax, total, payment_method, customer_sno, gst_enabled, business_id) 
                       VALUES (?, 'OPEN', 0, 0, 0, 0, '', ?, ?, ?)", 
                       [$table_no, $customer_sno, $gst_enabled, $business_id]);
                $order_id = db()->lastInsertId();

                // Add items to order
                foreach ($items as $item) {
                    $item_id = intval($item['item_id'] ?? 0);
                    $qty = max(1, intval($item['qty'] ?? 1));
                    
                    if ($item_id > 0) {
                        $item_data = one("SELECT * FROM items WHERE id = ? AND business_id = ?", [$item_id, $business_id]);
                        if ($item_data) {
                            query("INSERT INTO order_items(order_id, item_id, item_name, price, qty, business_id) VALUES (?,?,?,?,?,?)",
                                [$order_id, $item_data['id'], $item_data['name'], $item_data['price'], $qty, $business_id]);
                        }
                    }
                }

                // Recalculate totals
                recalc_order_totals($order_id);

                // Get the created order with items
                $created_order = one("SELECT o.*, c.name AS customer_name, c.mobile_no AS customer_mobile
                                     FROM orders o
                                     LEFT JOIN customer c ON o.customer_sno = c.sno
                                     WHERE o.id = ?", [$order_id]);
                $created_order['items'] = all_rows("SELECT * FROM order_items WHERE order_id = ?", [$order_id]);

                sendResponse(true, "Order created successfully", $created_order, 201);
            }
            
            // Add item to existing order
            elseif ($action === 'add_item') {
                $order_id = intval($input['order_id'] ?? 0);
                $item_id = intval($input['item_id'] ?? 0);
                $qty = max(1, intval($input['qty'] ?? 1));

                if ($order_id <= 0) {
                    sendResponse(false, "Invalid order ID", null, 400);
                }

                if ($item_id <= 0) {
                    sendResponse(false, "Invalid item ID", null, 400);
                }

                // Check if order exists and is open
                $order = one("SELECT status FROM orders WHERE id = ? AND business_id = ?", [$order_id, $business_id]);
                if (!$order) {
                    sendResponse(false, "Order not found", null, 404);
                }

                if ($order['status'] !== 'OPEN') {
                    sendResponse(false, "Cannot modify closed order", null, 400);
                }

                $item = one("SELECT * FROM items WHERE id = ? AND business_id = ?", [$item_id, $business_id]);
                if (!$item) {
                    sendResponse(false, "Item not found", null, 404);
                }

                query("INSERT INTO order_items(order_id, item_id, item_name, price, qty, business_id) VALUES (?,?,?,?,?,?)",
                    [$order_id, $item['id'], $item['name'], $item['price'], $qty, $business_id]);

                recalc_order_totals($order_id);
                sendResponse(true, "Item added to order successfully");
            }
            
            // Update order item quantity
            elseif ($action === 'update_item') {
                $oi_id = intval($input['oi_id'] ?? 0);
                $order_id = intval($input['order_id'] ?? 0);
                $qty = max(1, intval($input['qty'] ?? 1));

                if ($oi_id <= 0 || $order_id <= 0) {
                    sendResponse(false, "Invalid parameters", null, 400);
                }

                // Check if order is open
                $order = one("SELECT status FROM orders WHERE id = ? AND business_id = ?", [$order_id, $business_id]);
                if (!$order || $order['status'] !== 'OPEN') {
                    sendResponse(false, "Cannot modify closed order", null, 400);
                }

                query("UPDATE order_items SET qty = ? WHERE id = ?", [$qty, $oi_id]);
                recalc_order_totals($order_id);
                sendResponse(true, "Item quantity updated successfully");
            }
            
            // Remove item from order
            elseif ($action === 'remove_item') {
                $oi_id = intval($input['oi_id'] ?? 0);
                $order_id = intval($input['order_id'] ?? 0);

                if ($oi_id <= 0 || $order_id <= 0) {
                    sendResponse(false, "Invalid parameters", null, 400);
                }

                // Check if order is open
                $order = one("SELECT status FROM orders WHERE id = ? AND business_id = ?", [$order_id, $business_id]);
                if (!$order || $order['status'] !== 'OPEN') {
                    sendResponse(false, "Cannot modify closed order", null, 400);
                }

                query("DELETE FROM order_items WHERE id = ?", [$oi_id]);
                recalc_order_totals($order_id);
                sendResponse(true, "Item removed from order successfully");
            }
            
            // Apply discount to order
            elseif ($action === 'discount') {
                $order_id = intval($input['order_id'] ?? 0);
                $discount = (float)($input['discount'] ?? 0);

                if ($order_id <= 0) {
                    sendResponse(false, "Invalid order ID", null, 400);
                }

                // Check if order is open
                $order = one("SELECT status FROM orders WHERE id = ?", [$order_id]);
                if (!$order || $order['status'] !== 'OPEN') {
                    sendResponse(false, "Cannot modify closed order", null, 400);
                }

                query("UPDATE orders SET discount = ? WHERE id = ?", [$discount, $order_id]);
                recalc_order_totals($order_id);
                sendResponse(true, "Discount applied successfully");
            }
            
            // Process payment
            elseif ($action === 'pay') {
                $order_id = intval($input['order_id'] ?? 0);
                $payment_method = trim($input['payment_method'] ?? 'CASH');

                if ($order_id <= 0) {
                    sendResponse(false, "Invalid order ID", null, 400);
                }

                // Check if order exists and is open
                $order = one("SELECT status FROM orders WHERE id = ?", [$order_id]);
                if (!$order) {
                    sendResponse(false, "Order not found", null, 404);
                }

                if ($order['status'] !== 'OPEN') {
                    sendResponse(false, "Order is already processed", null, 400);
                }

                recalc_order_totals($order_id);
                query("UPDATE orders SET status = 'PAID', payment_method = ? WHERE id = ?", [$payment_method, $order_id]);

                sendResponse(true, "Payment processed successfully");
            }
            
            // Cancel order
            elseif ($action === 'cancel') {
                $order_id = intval($input['order_id'] ?? 0);

                if ($order_id <= 0) {
                    sendResponse(false, "Invalid order ID", null, 400);
                }

                // Check if order exists and is open
                $order = one("SELECT status FROM orders WHERE id = ?", [$order_id]);
                if (!$order) {
                    sendResponse(false, "Order not found", null, 404);
                }

                if ($order['status'] !== 'OPEN') {
                    sendResponse(false, "Cannot cancel processed order", null, 400);
                }

                query("UPDATE orders SET status = 'CANCELLED' WHERE id = ?", [$order_id]);
                sendResponse(true, "Order cancelled successfully");
            }
            
            else {
                sendResponse(false, "Invalid action", null, 400);
            }
            break;

        // ❌ Invalid Method
        default:
            sendResponse(false, "Method not allowed", null, 405);
    }
    
} catch (Exception $e) {
    sendResponse(false, "Server error: " . $e->getMessage(), null, 500);
}
?>
