<?php
header("Content-Type: application/json; charset=UTF-8");
header("Access-Control-Allow-Origin: *");
header("Access-Control-Allow-Methods: GET, POST, PUT, DELETE, OPTIONS");
header("Access-Control-Allow-Headers: Content-Type, Authorization, X-Requested-With");

// Handle preflight OPTIONS request
if ($_SERVER['REQUEST_METHOD'] == 'OPTIONS') {
    http_response_code(200);
    exit();
}

require_once __DIR__ . '/../db.php';
require_once __DIR__ . '/../session.php';

$method = $_SERVER['REQUEST_METHOD'];
$action = $_GET['action'] ?? '';
$input = json_decode(file_get_contents("php://input"), true);

// Helper function for response
function sendResponse($status, $message, $data = null, $code = 200) {
    http_response_code($code);
    echo json_encode([
        "status" => $status,
        "message" => $message,
        "data" => $data
    ]);
    exit;
}

// Auth guard for all non-OPTIONS requests
if (!validateApiToken()) {
    sendResponse(false, "Authentication required", null, 401);
}

$business_id = (int)($_SESSION['business_id'] ?? 0);

try {
    switch ($method) {

        // ✅ GET — Fetch all items or search items
        case 'GET':
            if ($action === 'list' || $action === '') {
                $search = trim($_GET['search'] ?? '');
                $category_id = $_GET['category_id'] ?? '';
                $active_only = $_GET['active_only'] ?? '1';
                
                $sql = "SELECT i.*, c.name AS category_name 
                        FROM items i 
                        LEFT JOIN categories c ON c.id = i.category_id 
                        WHERE i.business_id = ?";
                $params = [$business_id];
                
                if ($active_only === '1') {
                    $sql .= " AND i.is_active = 1";
                }
                
                if (!empty($search)) {
                    $sql .= " AND i.name LIKE ?";
                    $params[] = "%$search%";
                }
                
                if (!empty($category_id) && is_numeric($category_id)) {
                    $sql .= " AND i.category_id = ?";
                    $params[] = $category_id;
                }
                
                $sql .= " ORDER BY i.name";
                
                $items = all_rows($sql, $params);
                sendResponse(true, "Items fetched successfully", $items);
            }
            
            // Get single item
            elseif ($action === 'get') {
                $id = intval($_GET['id'] ?? 0);
                if ($id <= 0) {
                    sendResponse(false, "Invalid item ID", null, 400);
                }
                
                $item = one("SELECT i.*, c.name AS category_name 
                            FROM items i 
                            LEFT JOIN categories c ON c.id = i.category_id 
                            WHERE i.id = ? AND i.business_id = ?", [$id, $business_id]);
                if (!$item) {
                    sendResponse(false, "Item not found", null, 404);
                }
                
                sendResponse(true, "Item fetched successfully", $item);
            }
            
            // Search items for autocomplete
            elseif ($action === 'search') {
                $q = trim($_GET['q'] ?? '');
                if (empty($q)) {
                    sendResponse(true, "Search results", []);
                }
                
                $items = all_rows("SELECT id, name, price FROM items 
                                  WHERE is_active=1 AND business_id = ? AND name LIKE ? 
                                  ORDER BY name LIMIT 10", ["%$q%"]);
                
                // Adjust params to include business_id
                $items = all_rows("SELECT id, name, price FROM items 
                                  WHERE is_active=1 AND business_id = ? AND name LIKE ? 
                                  ORDER BY name LIMIT 10", [$business_id, "%$q%"]);
                sendResponse(true, "Search results", $items);
            }
            
            else {
                sendResponse(false, "Invalid action", null, 400);
            }
            break;

        // ✅ POST — Add a new item
        case 'POST':
            if ($action === 'create') {
                $name = trim($input['name'] ?? '');
                $price = (float)($input['price'] ?? 0);
                $sku = trim($input['sku'] ?? '');
                $category_id = !empty($input['category_id']) ? (int)$input['category_id'] : null;

                if (empty($name)) {
                    sendResponse(false, "Item name is required", null, 400);
                }

                if ($price <= 0) {
                    sendResponse(false, "Price must be greater than 0", null, 400);
                }

                // Check if item already exists
                $existing = one("SELECT id FROM items WHERE name = ? AND business_id = ?", [$name, $business_id]);
                if ($existing) {
                    sendResponse(false, "Item already exists", null, 409);
                }

                query("INSERT INTO items (name, price, sku, category_id, is_active, business_id) VALUES (?, ?, ?, ?, 1, ?)", 
                      [$name, $price, $sku, $category_id, $business_id]);
                $new_id = db()->lastInsertId();

                sendResponse(true, "Item added successfully", [
                    "id" => $new_id,
                    "name" => $name,
                    "price" => $price,
                    "sku" => $sku,
                    "category_id" => $category_id,
                    "is_active" => 1
                ], 201);
            }
            
            else {
                sendResponse(false, "Invalid action", null, 400);
            }
            break;

        // ✅ PUT — Update item
        case 'PUT':
            if ($action === 'update') {
                $id = intval($input['id'] ?? 0);
                $name = trim($input['name'] ?? '');
                $price = (float)($input['price'] ?? 0);
                $sku = trim($input['sku'] ?? '');
                $category_id = !empty($input['category_id']) ? (int)$input['category_id'] : null;
                $status = intval($input['is_active'] ?? 1);

                if ($id <= 0) {
                    sendResponse(false, "Invalid item ID", null, 400);
                }

                if (empty($name)) {
                    sendResponse(false, "Item name is required", null, 400);
                }

                if ($price <= 0) {
                    sendResponse(false, "Price must be greater than 0", null, 400);
                }

                // Check if item exists
                $existing = one("SELECT id FROM items WHERE id = ? AND business_id = ?", [$id, $business_id]);
                if (!$existing) {
                    sendResponse(false, "Item not found", null, 404);
                }

                // Check if name already exists for different item
                $nameExists = one("SELECT id FROM items WHERE name = ? AND id != ? AND business_id = ?", [$name, $id, $business_id]);
                if ($nameExists) {
                    sendResponse(false, "Item name already exists", null, 409);
                }

                query("UPDATE items SET name = ?, price = ?, sku = ?, category_id = ?, is_active = ? WHERE id = ? AND business_id = ?", 
                      [$name, $price, $sku, $category_id, $status, $id, $business_id]);

                sendResponse(true, "Item updated successfully", [
                    "id" => $id,
                    "name" => $name,
                    "price" => $price,
                    "sku" => $sku,
                    "category_id" => $category_id,
                    "is_active" => $status
                ]);
            }
            
            else {
                sendResponse(false, "Invalid action", null, 400);
            }
            break;

        // ✅ DELETE — Delete item
        case 'DELETE':
            if ($action === 'delete') {
                $id = intval($input['id'] ?? 0);

                if ($id <= 0) {
                    sendResponse(false, "Invalid item ID", null, 400);
                }

                // Check if item exists
                $existing = one("SELECT id FROM items WHERE id = ? AND business_id = ?", [$id, $business_id]);
                if (!$existing) {
                    sendResponse(false, "Item not found", null, 404);
                }

                // Check if item is used in any orders
                $hasOrders = one("SELECT oi.id FROM order_items oi JOIN orders o ON oi.order_id = o.id WHERE oi.item_id = ? AND o.business_id = ? LIMIT 1", [$id, $business_id]);
                if ($hasOrders) {
                    // Soft delete - just mark as inactive
                    query("UPDATE items SET is_active = 0 WHERE id = ? AND business_id = ?", [$id, $business_id]);
                    sendResponse(true, "Item deactivated successfully (used in orders)");
                } else {
                    // Hard delete
                    query("DELETE FROM items WHERE id = ? AND business_id = ?", [$id, $business_id]);
                    sendResponse(true, "Item deleted successfully");
                }
            }
            
            else {
                sendResponse(false, "Invalid action", null, 400);
            }
            break;

        // ❌ Invalid Method
        default:
            sendResponse(false, "Method not allowed", null, 405);
    }
    
} catch (Exception $e) {
    sendResponse(false, "Server error: " . $e->getMessage(), null, 500);
}
?>
