<?php
header("Content-Type: application/json; charset=UTF-8");
header("Access-Control-Allow-Origin: *");
header("Access-Control-Allow-Methods: GET, POST, PUT, DELETE, OPTIONS");
header("Access-Control-Allow-Headers: Content-Type, Authorization, X-Requested-With");

// Handle preflight OPTIONS request
if ($_SERVER['REQUEST_METHOD'] == 'OPTIONS') {
    http_response_code(200);
    exit();
}

require_once __DIR__ . '/../db.php';
require_once __DIR__ . '/../session.php';

$method = $_SERVER['REQUEST_METHOD'];
$action = $_GET['action'] ?? '';
$input = json_decode(file_get_contents("php://input"), true);

// 📌 Helper function for response
function sendResponse($status, $message, $data = null, $code = 200) {
    http_response_code($code);
    echo json_encode([
        "status" => $status,
        "message" => $message,
        "data" => $data
    ]);
    exit;
}

// Auth guard for all non-OPTIONS requests
if (!validateApiToken()) {
    sendResponse(false, "Authentication required", null, 401);
}

$business_id = (int)($_SESSION['business_id'] ?? 0);

try {
    switch ($method) {

        // ✅ GET — Fetch all categories
        case 'GET':
            if ($action === 'list' || $action === '') {
                $categories = all_rows("SELECT * FROM categories WHERE business_id = ? ORDER BY name", [$business_id]);
                sendResponse(true, "Categories fetched successfully", $categories);
            }
            
            // Get single category
            elseif ($action === 'get') {
                $id = intval($_GET['id'] ?? 0);
                if ($id <= 0) {
                    sendResponse(false, "Invalid category ID", null, 400);
                }
                
                $category = one("SELECT * FROM categories WHERE id = ? AND business_id = ?", [$id, $business_id]);
                if (!$category) {
                    sendResponse(false, "Category not found", null, 404);
                }
                
                sendResponse(true, "Category fetched successfully", $category);
            }
            
            else {
                sendResponse(false, "Invalid action", null, 400);
            }
            break;


        // ✅ POST — Add a new category
        case 'POST':
            if ($action === 'create') {
                $name = trim($input['name'] ?? '');

                if (empty($name)) {
                    sendResponse(false, "Category name is required", null, 400);
                }

                // Check if category already exists
                $existing = one("SELECT id FROM categories WHERE name = ?", [$name]);
                if ($existing) {
                    sendResponse(false, "Category already exists", null, 409);
                }

                query("INSERT INTO categories (name, is_active, business_id) VALUES (?, 1, ?)", [$name, $business_id]);
                $new_id = db()->lastInsertId();

                sendResponse(true, "Category added successfully", [
                    "id" => $new_id,
                    "name" => $name,
                    "is_active" => 1
                ], 201);
            }
            
            else {
                sendResponse(false, "Invalid action", null, 400);
            }
            break;


        // ✅ PUT — Update category
        case 'PUT':
            if ($action === 'update') {
                $id = intval($input['id'] ?? 0);
                $name = trim($input['name'] ?? '');
                $status = intval($input['is_active'] ?? 1);

                if ($id <= 0) {
                    sendResponse(false, "Invalid category ID", null, 400);
                }

                if (empty($name)) {
                    sendResponse(false, "Category name is required", null, 400);
                }

                // Check if category exists
                $existing = one("SELECT id FROM categories WHERE id = ? AND business_id = ?", [$id, $business_id]);
                if (!$existing) {
                    sendResponse(false, "Category not found", null, 404);
                }

                // Check if name already exists for different category
                $nameExists = one("SELECT id FROM categories WHERE name = ? AND id != ? AND business_id = ?", [$name, $id, $business_id]);
                if ($nameExists) {
                    sendResponse(false, "Category name already exists", null, 409);
                }

                query("UPDATE categories SET name = ?, is_active = ? WHERE id = ? AND business_id = ?", 
                      [$name, $status, $id, $business_id]);

                sendResponse(true, "Category updated successfully", [
                    "id" => $id,
                    "name" => $name,
                    "is_active" => $status
                ]);
            }
            
            else {
                sendResponse(false, "Invalid action", null, 400);
            }
            break;


        // ✅ DELETE — Delete a category
        case 'DELETE':
            if ($action === 'delete') {
                $id = intval($input['id'] ?? 0);

                if ($id <= 0) {
                    sendResponse(false, "Invalid category ID", null, 400);
                }

                // Check if category exists
                $existing = one("SELECT id FROM categories WHERE id = ? AND business_id = ?", [$id, $business_id]);
                if (!$existing) {
                    sendResponse(false, "Category not found", null, 404);
                }

                // Check if category has items
                $hasItems = one("SELECT id FROM items WHERE category_id = ? AND business_id = ? LIMIT 1", [$id, $business_id]);
                if ($hasItems) {
                    sendResponse(false, "Cannot delete category with existing items", null, 409);
                }

                query("DELETE FROM categories WHERE id = ? AND business_id = ?", [$id, $business_id]);
                sendResponse(true, "Category deleted successfully");
            }
            
            else {
                sendResponse(false, "Invalid action", null, 400);
            }
            break;


        // ❌ Invalid Method
        default:
            sendResponse(false, "Method not allowed", null, 405);
    }
    
} catch (Exception $e) {
    sendResponse(false, "Server error: " . $e->getMessage(), null, 500);
}
?>
