<?php
header("Content-Type: application/json; charset=UTF-8");
header("Access-Control-Allow-Origin: *");
header("Access-Control-Allow-Methods: GET, POST, PUT, DELETE, OPTIONS");
header("Access-Control-Allow-Headers: Content-Type, Authorization, X-Requested-With");

// Handle preflight OPTIONS request
if ($_SERVER['REQUEST_METHOD'] == 'OPTIONS') {
    http_response_code(200);
    exit();
}

require_once __DIR__ . '/../db.php';
require_once __DIR__ . '/../session.php';

$method = $_SERVER['REQUEST_METHOD'];
$action = $_GET['action'] ?? '';

// Helper function for JSON responses
function sendResponse($status, $message, $data = null, $code = 200) {
    http_response_code($code);
    echo json_encode([
        "status" => $status,
        "message" => $message,
        "data" => $data
    ]);
    exit;
}

// Helper function to generate JWT-like token (simple implementation)
function generateToken($business_id, $business_name = 'api_business') {
    return generateApiToken($business_id, $business_name, 'admin');
}

// Helper function to validate token
function validateToken($token) {
    try {
        $payload = json_decode(base64_decode($token), true);
        if (!$payload || $payload['exp'] < time()) {
            return false;
        }
        return $payload['business_id'];
    } catch (Exception $e) {
        return false;
    }
}

try {
    switch ($method) {
        
        // LOGIN
        case 'POST':
            if ($action === 'login') {
                $input = json_decode(file_get_contents('php://input'), true);
                
                $business_name = trim($input['business_name'] ?? '');
                $phone = trim($input['phone'] ?? '');
                $password = trim($input['password'] ?? '');
                
                if (empty($business_name) || empty($phone) || empty($password)) {
                    sendResponse(false, "Business name, phone, and password are required", null, 400);
                }
                
                // Fetch business details from database
                $business = one("SELECT * FROM settings WHERE business_name = ? AND phone = ?", [$business_name, $phone]);
                
                if ($business && password_verify($password, $business['password'])) {
                    // Check subscription status
                    $subscription = one("SELECT * FROM subscriptions WHERE restaurant_id = ? ORDER BY id DESC LIMIT 1", [$business['id']]);
                    
                    if ($subscription) {
                        $now = date('Y-m-d H:i:s');
                        if ($subscription['status'] === 'expired' || $subscription['expiry_date'] < $now) {
                            sendResponse(false, "Subscription expired. Please renew your subscription.", null, 403);
                        }
                    } else {
                        sendResponse(false, "No active subscription found. Please contact support.", null, 403);
                    }
                    
                    $token = generateToken($business['id'], $business['business_name']);
                    
                    // Create session
                    authenticateBusiness($business['id'], $business['business_name'], 'admin');
                    
                    sendResponse(true, "Login successful", [
                        "token" => $token,
                        "business" => [
                            "id" => $business['id'],
                            "business_name" => $business['business_name'],
                            "phone" => $business['phone'],
                            "role" => "admin"
                        ]
                    ]);
                } else {
                    sendResponse(false, "Invalid credentials", null, 401);
                }
            }
            
            // REGISTER
            elseif ($action === 'register') {
                $input = json_decode(file_get_contents('php://input'), true);
                
                $business_name = trim($input['business_name'] ?? '');
                $phone = trim($input['phone'] ?? '');
                $address = trim($input['address'] ?? '');
                $password = trim($input['password'] ?? '');
                
                if (empty($business_name) || empty($phone) || empty($address) || empty($password)) {
                    sendResponse(false, "All fields are required", null, 400);
                }
                
                // Check if business already exists
                $exists = one("SELECT * FROM settings WHERE phone = ?", [$phone]);
                if ($exists) {
                    sendResponse(false, "Business with this phone number already exists", null, 409);
                }
                
                $password_hash = password_hash($password, PASSWORD_BCRYPT);
                
                // Insert new business
                query("INSERT INTO settings (business_name, address, phone, password, gstin, gst_rate, gst_applicable) 
                       VALUES (?, ?, ?, ?, '', 5.00, 1)", 
                       [$business_name, $address, $phone, $password_hash]);
                
                $business_id = db()->lastInsertId();
                
                // Set 1-hour trial period (for demo)
                $start = date('Y-m-d H:i:s');
                $expiry = date('Y-m-d H:i:s', strtotime('+1 hour'));
                
                // Insert subscription record
                query("INSERT INTO subscriptions (restaurant_id, plan, start_date, expiry_date, status) 
                       VALUES (?, 'demo-1hour', ?, ?, 'active')", 
                       [$business_id, $start, $expiry]);
                
                sendResponse(true, "Business registered successfully", [
                    "business_id" => $business_id,
                    "business_name" => $business_name,
                    "trial_expires" => $expiry
                ], 201);
            }
            
            else {
                sendResponse(false, "Invalid action", null, 400);
            }
            break;
            
        // VALIDATE TOKEN
        case 'GET':
            if ($action === 'validate') {
                $headers = getallheaders();
                $auth_header = $headers['Authorization'] ?? '';
                
                if (!preg_match('/Bearer\s+(.*)$/i', $auth_header, $matches)) {
                    sendResponse(false, "No token provided", null, 401);
                }
                
                $token = $matches[1];
                $user_id = validateToken($token);
                
                if (!$user_id) {
                    sendResponse(false, "Invalid or expired token", null, 401);
                }
                
                sendResponse(true, "Token is valid", [
                    "user_id" => $user_id,
                    "valid" => true
                ]);
            }
            
            else {
                sendResponse(false, "Invalid action", null, 400);
            }
            break;
            
        // LOGOUT
        case 'PUT':
            if ($action === 'logout') {
                // In a real app, you'd invalidate the token on the server
                sendResponse(true, "Logged out successfully");
            }
            else {
                sendResponse(false, "Invalid action", null, 400);
            }
            break;
            
        default:
            sendResponse(false, "Method not allowed", null, 405);
    }
    
} catch (Exception $e) {
    sendResponse(false, "Server error: " . $e->getMessage(), null, 500);
}
?>
