<?php

require_once __DIR__ . '/../src/session.php';

// Redirect to login if not authenticated
if (!isAuthenticated()) {
    header("Location: login.php");
    exit;
}


ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);
error_reporting(E_ALL);

require_once __DIR__ . '/../src/config.php';
require_once __DIR__ . '/../src/db.php';
require_once __DIR__ . '/../src/helpers.php';
require_once __DIR__ . '/../vendor/autoload.php';

use PhpOffice\PhpSpreadsheet\IOFactory;

// Current business context
$current_business = getCurrentBusiness();
$business_id = (int)($current_business['business_id'] ?? 0);

function recalc_order_totals($order_id) {
    $order = one("SELECT discount FROM orders WHERE id=?", [$order_id]);
    $discount = (float)($order['discount'] ?? 0);

    $settings = one("SELECT gst_applicable, gst_rate FROM settings WHERE id=?", [$GLOBALS['business_id']]);
    $gst_enabled = (int)($settings['gst_applicable'] ?? 0);
    $gst_rate    = (float)($settings['gst_rate'] ?? 0);

    $ois = all_rows("SELECT * FROM order_items WHERE order_id=?", [$order_id]);
    $subtotal = 0.0;
    foreach ($ois as $oi) {
        $subtotal += $oi['price'] * $oi['qty'];
    }
    $taxable = max(0.0, $subtotal - $discount);
    $tax = 0.0;
    if ($gst_enabled && $gst_rate > 0) {
        $tax = round($taxable * ($gst_rate / 100.0), 2);
    }
    $total = round($taxable + $tax, 2);
    query("UPDATE orders 
              SET subtotal=?, discount=?, tax=?, total=? 
            WHERE id=?",
        [$subtotal, $discount, $tax, $total, $order_id]);
}
function is_order_open($order_id) {
    $order = one("SELECT status FROM orders WHERE id=?", [$order_id]);
    return $order && $order['status'] === 'OPEN';
}


query("INSERT INTO settings (id) VALUES (?) ON DUPLICATE KEY UPDATE id=id", [$business_id]);
$r = $_GET['r'] ?? '';

switch ($r) {
  case 'categories':
    render('categories');
    break;

  case 'categories/create':
    $name = trim($_POST['name'] ?? '');
    if ($name) {
      // Avoid duplicate names within the same business
      $existing = one("SELECT id FROM categories WHERE business_id = ? AND name = ?", [$business_id, $name]);
      if (!$existing) {
        query("INSERT INTO categories(name, business_id) VALUES (?, ?)", [$name, $business_id]);
      }
    }
    redirect(BASE_URL . '/public/?r=categories');
    break;

  case 'categories/set_status':
    $id = (int)($_GET['id'] ?? 0);
    $status = (int)($_GET['status'] ?? 1); 
    query("UPDATE categories SET is_active=? WHERE id=? AND business_id = ?", [$status, $id, $business_id]);

    if ($status == 0) {
        query("UPDATE items SET is_active=0 WHERE category_id=? AND business_id = ?", [$id, $business_id]);
    }
    redirect(BASE_URL . '/public/?r=categories');
    break;

  case 'categories/delete':
    $id = (int)($_GET['id'] ?? 0);
    query("DELETE FROM categories WHERE id=? AND business_id = ?", [$id, $business_id]);
    redirect(BASE_URL . '/public/?r=categories');
    break;

  case 'items':
    render('items');
    break;

  case 'items/create':
    $name = trim($_POST['name'] ?? '');
    $price = (float)($_POST['price'] ?? 0);
    $sku = trim($_POST['sku'] ?? '');
    $category_id = $_POST['category_id'] !== '' ? (int)$_POST['category_id'] : null;
    // Prevent duplicate item name per business
    $exists = one("SELECT id FROM items WHERE business_id = ? AND name = ?", [$business_id, $name]);
    if (!$exists) {
      query("INSERT INTO items(name,price,sku,category_id,business_id) VALUES (?,?,?,?,?)", [$name,$price,$sku,$category_id,$business_id]);
    }
    redirect(BASE_URL . '/public/?r=items');
    break;

  case 'items/import':
    if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_FILES['excel_file'])) {
        $file = $_FILES['excel_file']['tmp_name'];
        $spreadsheet = IOFactory::load($file);
        $sheet = $spreadsheet->getActiveSheet();
        foreach ($sheet->getRowIterator() as $row) {
            $cells = [];
            foreach ($row->getCellIterator() as $cell) {
                $cells[] = trim((string)$cell->getValue());
            }
            if (count($cells) >= 3 && strtolower($cells[0]) !== 'name') {
                $name  = $cells[0];
                $price = (float)$cells[1];
                $sku   = $cells[2];
                $cat   = null;
                if (isset($cells[3]) && $cells[3] !== '') {
                    $cat = (int)$cells[3];
                }
                query("INSERT INTO items(name,price,sku,category_id,business_id) VALUES (?,?,?,?,?)", [$name,$price,$sku,$cat,$business_id]);
            }
        }
    }
    redirect(BASE_URL . '/public/?r=items');
    break;

  case 'items/set_status':
    $id = (int)($_GET['id'] ?? 0);
    $status = (int)($_GET['status'] ?? 1); 
    query("UPDATE items SET is_active=? WHERE id=? AND business_id = ?", [$status, $id, $business_id]);
    redirect(BASE_URL . '/public/?r=items');
    break;

  case 'items/delete':
    $id = (int)($_GET['id'] ?? 0);
    query("DELETE FROM items WHERE id=? AND business_id = ?", [$id, $business_id]);
    redirect(BASE_URL . '/public/?r=items');
    break;

  case 'orders':
    render('orders');
    break;

  case 'orders/new':
    if ($_SERVER['REQUEST_METHOD'] === 'POST') {
        $order_type = trim($_POST['order_type'] ?? 'table');
        $table_no_post = trim($_POST['table_no'] ?? '');
        $customer_name = trim($_POST['customer_name'] ?? '');
        $customer_mobile = trim($_POST['customer_mobile'] ?? '');

        if (strtolower($order_type) === 'takeaway') {
            $table_no = 'Takeaway';
        } else {
            $table_no = $table_no_post;
        }

        query("INSERT INTO customer (name, mobile_no, business_id) VALUES (?, ?, ?)", 
              [$customer_name, $customer_mobile, $business_id]);
        $customer_sno = db()->lastInsertId();

        // 🔹 get global gst setting
        $settings = one("SELECT gst_applicable FROM settings WHERE id=?", [$business_id]);
        $gst_enabled = (int)($settings['gst_applicable'] ?? 0);

        query("INSERT INTO orders (table_no, status, subtotal, discount, tax, total, payment_method, customer_sno, gst_enabled, business_id) 
               VALUES (?, 'OPEN', 0, 0, 0, 0, '', ?, ?, ?)", 
               [$table_no, $customer_sno, $gst_enabled, $business_id]);
        $id = db()->lastInsertId();

        redirect(BASE_URL . '/public/?r=orders/show&id=' . $id);
        exit;
    }
    render('order_new');
    break;

  case 'orders/show':
    render('order_show');
    break;

  case 'orders/add_item':
    $order_id = (int)$_POST['order_id'];
    if (!is_order_open($order_id)) {
        redirect(BASE_URL . '/public/?r=orders/show&id=' . $order_id);
        break;
    }
    $item = one("SELECT * FROM items WHERE id=? AND business_id = ?", [(int)$_POST['item_id'], $business_id]);
    $qty  = max(1, (int)$_POST['qty']);
    if ($item) {
      query("INSERT INTO order_items(order_id,item_id,item_name,price,qty,business_id) VALUES (?,?,?,?,?,?)",
        [$order_id,$item['id'],$item['name'],$item['price'],$qty,$business_id]);
      recalc_order_totals($order_id);
    }
    redirect(BASE_URL . '/public/?r=orders/show&id=' . $order_id);
    break;

  case 'orders/update_item':
    $oi_id = (int)$_POST['oi_id'];
    $order_id = (int)$_POST['order_id'];
    if (!is_order_open($order_id)) {
        redirect(BASE_URL . '/public/?r=orders/show&id=' . $order_id);
        break;
    }
    $qty = max(1, (int)$_POST['qty']);
    query("UPDATE order_items SET qty=? WHERE id=?", [$qty,$oi_id]);
    recalc_order_totals($order_id);
    redirect(BASE_URL . '/public/?r=orders/show&id=' . $order_id);
    break;

  case 'orders/remove_item':
    $oi_id = (int)$_GET['oi_id'];
    $order_id = (int)$_GET['order_id'];
    if (!is_order_open($order_id)) {
        redirect(BASE_URL . '/public/?r=orders/show&id=' . $order_id);
        break;
    }
    query("DELETE FROM order_items WHERE id=?", [$oi_id]);
    recalc_order_totals($order_id);
    redirect(BASE_URL . '/public/?r=orders/show&id=' . $order_id);
    break;

  case 'orders/discount':
    $order_id = (int)$_POST['order_id'];
    if (!is_order_open($order_id)) {
        redirect(BASE_URL . '/public/?r=orders/show&id=' . $order_id);
        break;
    }
    $discount = (float)($_POST['discount'] ?? 0);
    query("UPDATE orders SET discount=? WHERE id=? AND business_id = ?", [$discount, $order_id, $business_id]);
    recalc_order_totals($order_id);
    redirect(BASE_URL . '/public/?r=orders/show&id=' . $order_id);
    break;

  case 'orders/toggle_gst':
    $order_id = (int)$_POST['order_id'];
    if (!is_order_open($order_id)) {
        redirect(BASE_URL . '/public/?r=orders/show&id=' . $order_id);
        break;
    }
    $gst_enabled = isset($_POST['gst_enabled']) ? 1 : 0;
    query("UPDATE orders SET gst_enabled=? WHERE id=? AND business_id = ?", [$gst_enabled, $order_id, $business_id]);
    recalc_order_totals($order_id);
    redirect(BASE_URL . '/public/?r=orders/show&id=' . $order_id);
    break;

  case 'orders/pay':
    $order_id = $_POST['order_id'];
    $method   = $_POST['payment_method'] ?? 'CASH';
    recalc_order_totals($order_id);
    query("UPDATE orders SET status='PAID', payment_method=? WHERE id=? AND business_id = ?", [$method, $order_id, $business_id]);
    redirect(BASE_URL . '/public/?r=orders/show&id=' . $order_id);
    break;

  case 'orders/cancel':
    $order_id = (int)$_GET['id'];
    query("UPDATE orders SET status='CANCELLED' WHERE id=? AND business_id = ?", [$order_id, $business_id]);
    redirect(BASE_URL . '/public/?r=orders/show&id=' . $order_id);
    break;

  case 'orders/receipt':
    require __DIR__ . '/../src/views/receipt.php';
    break;

  case 'reports':
    render('reports');
    break;

  case 'reports.csv':
    $from = $_GET['from'] ?? date('Y-m-01');
    $to   = $_GET['to']   ?? date('Y-m-d');

    $rows = all_rows("
        SELECT o.id, o.created_at, o.table_no, o.total,
               c.name AS customer_name, c.mobile_no AS customer_mobile
        FROM orders o
        LEFT JOIN customer c ON o.customer_sno = c.sno
        WHERE o.business_id = ? AND DATE(o.created_at) BETWEEN ? AND ?
          AND o.status='PAID'
        ORDER BY o.created_at DESC
    ", [$business_id,$from,$to]);

    header('Content-Type: text/csv');
    header('Content-Disposition: attachment; filename="reports.csv"');
    $out = fopen('php://output', 'w');

    // Add headers
    fputcsv($out, ['ID','Customer Name','Customer Mobile','Date','Table','Total']);

    foreach ($rows as $r) {
        fputcsv($out, [
            $r['id'],
            $r['customer_name'],
            $r['customer_mobile'],
            $r['created_at'],
            $r['table_no'],
            $r['total']
        ]);
    }

    fclose($out);
    exit;
    break;

  case 'settings':
    render('settings');
    break;

  case 'settings/save':
    $business_name = trim($_POST['business_name'] ?? '');
    $phone = trim($_POST['phone'] ?? '');
    $address = trim($_POST['address'] ?? '');
    $gstin = trim($_POST['gstin'] ?? '');
    $gst_rate = (float)($_POST['gst_rate'] ?? 0);
    $gst_applicable = (int)($_POST['gst_applicable'] ?? 0);

    query("UPDATE settings 
              SET business_name=?, phone=?, address=?, gstin=?, gst_rate=?, gst_applicable=? 
            WHERE id=?",
          [$business_name,$phone,$address,$gstin,$gst_rate,$gst_applicable,$business_id]);

    redirect(BASE_URL . '/public/?r=settings');
    break;

  case 'api/items':
    $q = trim($_GET['q'] ?? '');
    $items = all_rows("SELECT id,name,price FROM items WHERE business_id = ? AND is_active=1 AND name LIKE ? ORDER BY name LIMIT 10", [$business_id, "%$q%"]);
    header('Content-Type: application/json');
    echo json_encode($items);
    exit;
    break;

  default:
    render('dashboard');
}
