<?php
header("Content-Type: application/json; charset=UTF-8");
header("Access-Control-Allow-Origin: *");
header("Access-Control-Allow-Methods: POST, OPTIONS");
header("Access-Control-Allow-Headers: Content-Type, Authorization, X-Requested-With");

// Handle preflight OPTIONS request
if ($_SERVER['REQUEST_METHOD'] == 'OPTIONS') {
    http_response_code(200);
    exit();
}

require_once __DIR__ . '/../db.php';
require_once __DIR__ . '/../session.php';

$method = $_SERVER['REQUEST_METHOD'];
$input = json_decode(file_get_contents("php://input"), true);

function sendResponse($status, $message, $data = null, $code = 200) {
    http_response_code($code);
    echo json_encode([
        "status" => $status,
        "message" => $message,
        "data" => $data
    ]);
    exit;
}

// Auth guard
if (!validateApiToken()) {
    sendResponse(false, "Authentication required", null, 401);
}

$business_id = (int)($_SESSION['business_id'] ?? 0);

try {
    if ($method !== 'POST') {
        sendResponse(false, "Method not allowed", null, 405);
    }

    // Backward-compatible shim that forwards to Orders API create semantics
    $order_type = trim($input['order_type'] ?? 'table');
    $table_no = trim($input['table_no'] ?? '');
    $customer_name = trim($input['customer_name'] ?? '');
    $customer_mobile = trim($input['customer_mobile'] ?? '');
    $items = $input['items'] ?? [];

    if (empty($customer_name)) {
        sendResponse(false, "Customer name is required", null, 400);
    }
    if (empty($items) || !is_array($items)) {
        sendResponse(false, "At least one item is required", null, 400);
    }

    if (strtolower($order_type) === 'takeaway') {
        $table_no = 'Takeaway';
    }

    // Create customer
    query("INSERT INTO customer (name, mobile_no, business_id) VALUES (?, ?, ?)", [$customer_name, $customer_mobile, $business_id]);
    $customer_sno = last_id();

    // Get GST flag
    $settings = one("SELECT gst_applicable FROM settings WHERE id=1");
    $gst_enabled = (int)($settings['gst_applicable'] ?? 0);

    // Create order
    query("INSERT INTO orders (table_no, status, subtotal, discount, tax, total, payment_method, customer_sno, gst_enabled, business_id) VALUES (?, 'OPEN', 0, 0, 0, 0, '', ?, ?, ?)", [$table_no, $customer_sno, $gst_enabled, $business_id]);
    $order_id = last_id();

    // Add items
    foreach ($items as $it) {
        $item_id = intval($it['item_id'] ?? $it['id'] ?? 0);
        $qty = max(1, intval($it['qty'] ?? 1));
        if ($item_id > 0) {
            $item_data = one("SELECT * FROM items WHERE id = ?", [$item_id]);
            if ($item_data) {
                query("INSERT INTO order_items(order_id, item_id, item_name, price, qty, business_id) VALUES (?,?,?,?,?,?)", [$order_id, $item_data['id'], $item_data['name'], $item_data['price'], $qty, $business_id]);
            }
        }
    }

    // Recalc totals (copy of helper from orders.php)
    $order = one("SELECT discount FROM orders WHERE id=?", [$order_id]);
    $discount = (float)($order['discount'] ?? 0);
    $settings2 = one("SELECT gst_applicable, gst_rate FROM settings WHERE id=1");
    $gst_enabled2 = (int)($settings2['gst_applicable'] ?? 0);
    $gst_rate = (float)($settings2['gst_rate'] ?? 0);
    $ois = all_rows("SELECT * FROM order_items WHERE order_id=?", [$order_id]);
    $subtotal = 0.0;
    foreach ($ois as $oi) { $subtotal += $oi['price'] * $oi['qty']; }
    $taxable = max(0.0, $subtotal - $discount);
    $tax = 0.0; if ($gst_enabled2 && $gst_rate > 0) { $tax = round($taxable * ($gst_rate / 100.0), 2); }
    $total = round($taxable + $tax, 2);
    query("UPDATE orders SET subtotal=?, discount=?, tax=?, total=? WHERE id=?", [$subtotal, $discount, $tax, $total, $order_id]);

    $created = one("SELECT o.*, c.name AS customer_name, c.mobile_no AS customer_mobile FROM orders o LEFT JOIN customer c ON o.customer_sno = c.sno WHERE o.id = ?", [$order_id]);
    $created['items'] = all_rows("SELECT * FROM order_items WHERE order_id = ?", [$order_id]);

    sendResponse(true, "Order created successfully", $created, 201);
} catch (Exception $e) {
    sendResponse(false, "Server error: " . $e->getMessage(), null, 500);
}
?>
