<?php
header("Content-Type: application/json; charset=UTF-8");
header("Access-Control-Allow-Origin: *");
header("Access-Control-Allow-Methods: GET, OPTIONS");
header("Access-Control-Allow-Headers: Content-Type, Authorization, X-Requested-With");

// Handle preflight OPTIONS request
if ($_SERVER['REQUEST_METHOD'] == 'OPTIONS') {
    http_response_code(200);
    exit();
}

require_once __DIR__ . "/../db.php";
require_once __DIR__ . '/../session.php';

$method = $_SERVER['REQUEST_METHOD'];
$action = $_GET['action'] ?? '';

// Helper function for response
function sendResponse($status, $message, $data = null, $code = 200) {
    http_response_code($code);
    echo json_encode([
        "status" => $status,
        "message" => $message,
        "data" => $data
    ]);
    exit;
}

// Auth guard for all non-OPTIONS requests
if (!validateApiToken()) {
    sendResponse(false, "Authentication required", null, 401);
}

$business_id = (int)($_SESSION['business_id'] ?? 0);

try {
    if ($method === 'GET') {
        
        if ($action === 'stats' || $action === '') {
            // Get today's date range
            $today_start = date('Y-m-d 00:00:00');
            $today_end = date('Y-m-d 23:59:59');
            
            // Get current month date range
            $month_start = date('Y-m-01 00:00:00');
            $month_end = date('Y-m-d 23:59:59');
            
            // Today's stats
            $today_stats = one("SELECT 
                                COUNT(*) as total_orders,
                                SUM(CASE WHEN status = 'PAID' THEN 1 ELSE 0 END) as paid_orders,
                                SUM(CASE WHEN status = 'OPEN' THEN 1 ELSE 0 END) as open_orders,
                                SUM(CASE WHEN status = 'PAID' THEN total ELSE 0 END) as total_revenue,
                                SUM(CASE WHEN status = 'PAID' THEN subtotal ELSE 0 END) as total_subtotal,
                                SUM(CASE WHEN status = 'PAID' THEN discount ELSE 0 END) as total_discount,
                                SUM(CASE WHEN status = 'PAID' THEN tax ELSE 0 END) as total_tax
                             FROM orders 
                             WHERE business_id = ? AND created_at BETWEEN ? AND ?", [$business_id, $today_start, $today_end]);
            
            // Month's stats
            $month_stats = one("SELECT 
                                COUNT(*) as total_orders,
                                SUM(CASE WHEN status = 'PAID' THEN 1 ELSE 0 END) as paid_orders,
                                SUM(CASE WHEN status = 'OPEN' THEN 1 ELSE 0 END) as open_orders,
                                SUM(CASE WHEN status = 'PAID' THEN total ELSE 0 END) as total_revenue,
                                SUM(CASE WHEN status = 'PAID' THEN subtotal ELSE 0 END) as total_subtotal,
                                SUM(CASE WHEN status = 'PAID' THEN discount ELSE 0 END) as total_discount,
                                SUM(CASE WHEN status = 'PAID' THEN tax ELSE 0 END) as total_tax
                             FROM orders 
                             WHERE business_id = ? AND created_at BETWEEN ? AND ?", [$business_id, $month_start, $month_end]);
            
            // Recent orders
            $open_orders = all_rows("SELECT o.*, c.name AS customer_name, c.mobile_no AS customer_mobile
                                    FROM orders o
                                    LEFT JOIN customer c ON o.customer_sno = c.sno
                                    WHERE o.status='OPEN' AND o.business_id = ? 
                                    ORDER BY o.created_at DESC LIMIT 10", [$business_id]);
            
            $recent_paid = all_rows("SELECT o.*, c.name AS customer_name, c.mobile_no AS customer_mobile
                                    FROM orders o
                                    LEFT JOIN customer c ON o.customer_sno = c.sno
                                    WHERE o.status='PAID' AND o.business_id = ? 
                                    ORDER BY o.created_at DESC LIMIT 10", [$business_id]);
            
            // Top selling items (this month)
            $top_items = all_rows("SELECT 
                                    oi.item_name, 
                                    SUM(oi.qty) as total_qty,
                                    SUM(oi.price * oi.qty) as total_amount,
                                    COUNT(DISTINCT oi.order_id) as order_count
                                  FROM order_items oi
                                  INNER JOIN orders o ON oi.order_id = o.id
                                  WHERE o.status = 'PAID' AND o.business_id = ? 
                                    AND o.created_at BETWEEN ? AND ?
                                  GROUP BY oi.item_name
                                  ORDER BY total_qty DESC
                                  LIMIT 10", [$business_id, $month_start, $month_end]);
            
            // Payment method breakdown (today)
            $payment_methods = all_rows("SELECT 
                                          payment_method,
                                          COUNT(*) as order_count,
                                          SUM(total) as total_amount
                                        FROM orders 
                                        WHERE status = 'PAID' AND business_id = ?
                                          AND created_at BETWEEN ? AND ?
                                        GROUP BY payment_method
                                        ORDER BY total_amount DESC", [$business_id, $today_start, $today_end]);
            
            $dashboard_data = [
                "today" => $today_stats,
                "month" => $month_stats,
                "open_orders" => $open_orders,
                "recent_paid" => $recent_paid,
                "top_items" => $top_items,
                "payment_methods" => $payment_methods
            ];
            
            sendResponse(true, "Dashboard data fetched successfully", $dashboard_data);
        }
        
        else {
            sendResponse(false, "Invalid action", null, 400);
        }
    }
    
    else {
        sendResponse(false, "Method not allowed", null, 405);
    }
    
} catch (Exception $e) {
    sendResponse(false, "Server error: " . $e->getMessage(), null, 500);
}
?>
