<?php
header("Content-Type: application/json; charset=UTF-8");
header("Access-Control-Allow-Origin: *");
header("Access-Control-Allow-Methods: GET, POST, PUT, DELETE, OPTIONS");
header("Access-Control-Allow-Headers: Content-Type, Authorization, X-Requested-With");

// Handle preflight OPTIONS request
if ($_SERVER['REQUEST_METHOD'] == 'OPTIONS') {
    http_response_code(200);
    exit();
}

require_once __DIR__ . '/../db.php';
require_once __DIR__ . '/../session.php';

$method = $_SERVER['REQUEST_METHOD'];
$action = $_GET['action'] ?? '';
$input = json_decode(file_get_contents("php://input"), true);

// Helper function for response
function sendResponse($status, $message, $data = null, $code = 200) {
    http_response_code($code);
    echo json_encode([
        "status" => $status,
        "message" => $message,
        "data" => $data
    ]);
    exit;
}

// Auth guard for all non-OPTIONS requests
if (!validateApiToken()) {
    sendResponse(false, "Authentication required", null, 401);
}

$business_id = (int)($_SESSION['business_id'] ?? 0);

try {
    switch ($method) {

        // ✅ GET — Fetch customers
        case 'GET':
            if ($action === 'list' || $action === '') {
                $search = trim($_GET['search'] ?? '');
                $limit = intval($_GET['limit'] ?? 100);
                
                $sql = "SELECT c.*, 
                               COUNT(o.id) as total_orders,
                               SUM(CASE WHEN o.status = 'PAID' THEN o.total ELSE 0 END) as total_spent,
                               MAX(o.created_at) as last_order_date
                        FROM customer c
                        LEFT JOIN orders o ON c.sno = o.customer_sno
                        WHERE c.business_id = ?";
                $params = [$business_id];
                
                if (!empty($search)) {
                    $sql .= " AND (c.name LIKE ? OR c.mobile_no LIKE ?)";
                    $params[] = "%$search%";
                    $params[] = "%$search%";
                }
                
                $sql .= " GROUP BY c.sno ORDER BY c.sno DESC LIMIT ?";
                $params[] = $limit;
                
                $customers = all_rows($sql, $params);
                sendResponse(true, "Customers fetched successfully", $customers);
            }
            
            // Get single customer
            elseif ($action === 'get') {
                $id = intval($_GET['id'] ?? 0);
                if ($id <= 0) {
                    sendResponse(false, "Invalid customer ID", null, 400);
                }
                
                $customer = one("SELECT c.*, 
                                       COUNT(o.id) as total_orders,
                                       SUM(CASE WHEN o.status = 'PAID' THEN o.total ELSE 0 END) as total_spent,
                                       MAX(o.created_at) as last_order_date
                                FROM customer c
                                LEFT JOIN orders o ON c.sno = o.customer_sno
                                WHERE c.sno = ? AND c.business_id = ?
                                GROUP BY c.sno", [$id, $business_id]);
                
                if (!$customer) {
                    sendResponse(false, "Customer not found", null, 404);
                }
                
                // Get customer's order history
                $orders = all_rows("SELECT o.*, 
                                          COUNT(oi.id) as item_count
                                   FROM orders o
                                   LEFT JOIN order_items oi ON o.id = oi.order_id
                                   WHERE o.customer_sno = ? AND o.business_id = ?
                                   GROUP BY o.id
                                   ORDER BY o.created_at DESC
                                   LIMIT 20", [$id, $business_id]);
                
                $customer['orders'] = $orders;
                sendResponse(true, "Customer fetched successfully", $customer);
            }
            
            // Search customers by mobile or name
            elseif ($action === 'search') {
                $q = trim($_GET['q'] ?? '');
                if (empty($q)) {
                    sendResponse(true, "Search results", []);
                }
                
                $customers = all_rows("SELECT sno, name, mobile_no FROM customer 
                                      WHERE business_id = ? AND (name LIKE ? OR mobile_no LIKE ?) 
                                      ORDER BY name LIMIT 10", [$business_id, "%$q%", "%$q%"]);
                sendResponse(true, "Search results", $customers);
            }
            
            else {
                sendResponse(false, "Invalid action", null, 400);
            }
            break;

        // ✅ POST — Add a new customer
        case 'POST':
            if ($action === 'create') {
                $name = trim($input['name'] ?? '');
                $mobile_no = trim($input['mobile_no'] ?? '');

                if (empty($name)) {
                    sendResponse(false, "Customer name is required", null, 400);
                }

                if (empty($mobile_no)) {
                    sendResponse(false, "Mobile number is required", null, 400);
                }

                // Check if customer with same mobile already exists
                $existing = one("SELECT sno FROM customer WHERE mobile_no = ? AND business_id = ?", [$mobile_no, $business_id]);
                if ($existing) {
                    sendResponse(false, "Customer with this mobile number already exists", null, 409);
                }

                query("INSERT INTO customer (name, mobile_no, business_id) VALUES (?, ?, ?)", [$name, $mobile_no, $business_id]);
                $new_id = db()->lastInsertId();

                sendResponse(true, "Customer added successfully", [
                    "sno" => $new_id,
                    "name" => $name,
                    "mobile_no" => $mobile_no
                ], 201);
            }
            
            else {
                sendResponse(false, "Invalid action", null, 400);
            }
            break;

        // ✅ PUT — Update customer
        case 'PUT':
            if ($action === 'update') {
                $sno = intval($input['sno'] ?? 0);
                $name = trim($input['name'] ?? '');
                $mobile_no = trim($input['mobile_no'] ?? '');

                if ($sno <= 0) {
                    sendResponse(false, "Invalid customer ID", null, 400);
                }

                if (empty($name)) {
                    sendResponse(false, "Customer name is required", null, 400);
                }

                if (empty($mobile_no)) {
                    sendResponse(false, "Mobile number is required", null, 400);
                }

                // Check if customer exists
                $existing = one("SELECT sno FROM customer WHERE sno = ? AND business_id = ?", [$sno, $business_id]);
                if (!$existing) {
                    sendResponse(false, "Customer not found", null, 404);
                }

                // Check if mobile number already exists for different customer
                $mobileExists = one("SELECT sno FROM customer WHERE mobile_no = ? AND sno != ? AND business_id = ?", [$mobile_no, $sno, $business_id]);
                if ($mobileExists) {
                    sendResponse(false, "Mobile number already exists for another customer", null, 409);
                }

                query("UPDATE customer SET name = ?, mobile_no = ? WHERE sno = ? AND business_id = ?", 
                      [$name, $mobile_no, $sno, $business_id]);

                sendResponse(true, "Customer updated successfully", [
                    "sno" => $sno,
                    "name" => $name,
                    "mobile_no" => $mobile_no
                ]);
            }
            
            else {
                sendResponse(false, "Invalid action", null, 400);
            }
            break;

        // ✅ DELETE — Delete customer
        case 'DELETE':
            if ($action === 'delete') {
                $sno = intval($input['sno'] ?? 0);

                if ($sno <= 0) {
                    sendResponse(false, "Invalid customer ID", null, 400);
                }

                // Check if customer exists
                $existing = one("SELECT sno FROM customer WHERE sno = ? AND business_id = ?", [$sno, $business_id]);
                if (!$existing) {
                    sendResponse(false, "Customer not found", null, 404);
                }

                // Check if customer has orders
                $hasOrders = one("SELECT id FROM orders WHERE customer_sno = ? AND business_id = ? LIMIT 1", [$sno, $business_id]);
                if ($hasOrders) {
                    sendResponse(false, "Cannot delete customer with existing orders", null, 409);
                }

                query("DELETE FROM customer WHERE sno = ? AND business_id = ?", [$sno, $business_id]);
                sendResponse(true, "Customer deleted successfully");
            }
            
            else {
                sendResponse(false, "Invalid action", null, 400);
            }
            break;

        // ❌ Invalid Method
        default:
            sendResponse(false, "Method not allowed", null, 405);
    }
    
} catch (Exception $e) {
    sendResponse(false, "Server error: " . $e->getMessage(), null, 500);
}
?>
